unit verResU;

// Copyright  2000 by Ziff Davis Media, Inc.
// Written by Neil J. Rubenking

// Objects for decoding Version resource data

interface
USES Windows, SysUtils, Classes;
type
  TVerLangObj = class(TObject)
  private
    fLanguage : String;   // Language/charset string, e.g. 040904E4
    fList     : TStringList; // name=value list
    function GetValue(const Name: string): string;
    function Rating : Integer;
  public
    constructor Create(P : PByteArray; fUnicode : Boolean);
    destructor Destroy; override;
    property Language : String read fLanguage;
  end;

  TVerInfoObj = class(TObject)
  private
    fStrLangs   : TStringList;
    fVarLangs   : TStringList;
    fVFFI       : TVSFixedFileInfo;
    fStatus     : Integer;
    fUnicode    : Boolean;     // true if data is unicode
    function GetCount: Integer;
    function GetFileVersion: String;
    function GetProductVersion: String;
    function GetString(Index: Integer): string;
    function GetValue(const Name: string): string;
    function GetFileType: Integer;
    function GetFileTypeS: String;
    function GetLangCount: Integer;
    function GetLanguage(Index: Integer): string;
    function GetLangValue(const Lang, Name: string): string;
    function GetLangString(IndexL, IndexS: Integer): string;
  public
    constructor Create(const S : String);
    destructor Destroy; override;
    function LangProb(VAR S : String) : Boolean;
    // read-only properties
    property Count          : Integer read GetCount;
    property FileType       : Integer read GetFileType;
    property FileTypeS      : String read GetFileTypeS;
    property FileVersion    : String read GetFileVersion;
    property LangCount      : Integer read GetLangCount;
    property ProductVersion : String read GetProductVersion;
    property Status         : Integer read fStatus;
    property Unicode        : Boolean read fUnicode;
    property VFFI           : TVSFixedFileInfo read fVFFI;
    // read-only indexed properties
    property Languages[Index : Integer] : string read GetLanguage;
    property LangStrings[IndexL, IndexS : Integer] : string
      read GetLangString;
    property LangValues[const Lang, Name: string] : string read
      GetLangValue;
    property Strings[Index : Integer]   : string read GetString;
    property Values[const Name: string] : string read GetValue;
  end;

const
  vio_NoInfo    = 4; // version info size = 0 - no info
  vio_Failed    = 3; // failed to get version info
  vio_Invalid   = 2; // invalid - not VS_VERSION_INFO
  vio_NoStrings = 1; // valid, but contains no StringFileInfo
  vio_OK        = 0; // just dandy

implementation
VAR
  IsWinNT  : Boolean; // true if VER_PLATFORM_WIN32_NT
  PrefLang : String;  // 040904E4

function Pad32(I : Integer) : Integer;
// The version info structures are not true structures. Their field
//   layout is not fixed. Various fields are separated by "padding"
//   to bring them to a 32-bit boundary. This function serves to
//   skip the padding.
begin
  CASE I MOD 4 OF
    1 :  Result := I + 3;
    2 :  Result := I + 2;
    3 :  Result := I + 1;
    ELSE Result := I;
  END;
end;

{ TVerInfoObj }

constructor TVerInfoObj.Create(const S: String);
VAR
  verInfoSize : Integer;     // size of version info block
  DummyGets0  : DWORD;       // required by GetFileVersionInfoSize
  VerBuff     : PWordArray;  // buffer for version info block
  VerLeng     : Word;        // VS_VERSION_INFO.wLength
  ValLeng     : Word;        // VS_VERSION_INFO.wValueLength
  VerKey      : String;      // VS_VERSION_INFO.szKey
  ChdBuff     : PWordArray;  // StringFileInfo structure
  ChdLeng     : Word;        // StringFileInfo.wLength
  ChdKey      : String;      // StringFileInfo.szKey
  VarBuff     : PWordArray;  // VarFileInfo structure
  VarLeng     : Word;        // VarFileInfo.wLength
  VarKey      : String;      // VarFileInfo.szKey
  rOfs        : Integer;     // offset within varFileInfo block
  vOfs        : Integer;     // offset within main block
  cOfs        : Integer;     // offset within StringFileInfo block
  TabLeng     : Integer;     // StringTable length
  VarPs       : PDWORD;      // language codes in VarFileInfo
  VL          : TVerLangObj;
begin
  Inherited Create;
  // Initialize private data members
  fStrLangs   := TStringList.Create;
  fVarLangs   := TStringList.Create;
  fStatus     := vio_NoInfo;
  // Get size of  version info
  verInfoSize := GetFileVersionInfoSize(PChar(S), DummyGets0);
  IF VerInfoSize = 0 THEN
    Exit;
  // Allocate buffer
  GetMem(VerBuff, verInfoSize);
  try
    fStatus := vio_Failed;
    IF NOT GetFileVersionInfoA(PChar(S), 0, verInfoSize, VerBuff) THEN
      Exit;
    VerLeng := VerBuff^[0]; // first Word of VS_VERSION_INFO
    ValLeng := VerBuff^[1]; // second Word of VS_VERSION_INFO
    fUnicode := WideCharToString(PWideChar(@VerBuff^[3])) =
      'VS_VERSION_INFO';
    IF fUnicode THEN // Key is UNICODE at offset 6
      begin
        VerKey := WideCharToString(PWideChar(@VerBuff^[3]));
        vOfs   := Pad32(6 + succ(length(VerKey))*2);
      end
    ELSE            // Key is ANSI at offset 4
      begin
        VerKey := StrPas(PChar(@VerBuff^[2]));
        vOfs   := Pad32(4 + succ(length(VerKey)));
      end;
    fStatus := vio_Invalid;
    IF VerKey <> 'VS_VERSION_INFO' THEN
      Exit;
    // Copy VS_FIXEDFILEINFO, if available
    FillChar(fVFFI, SizeOf(fVFFI), 0);
    IF ValLeng > 0 THEN
      fVFFI := PVSFixedFileInfo(@VerBuff^[vOfs DIV 2])^;
    vOfs       := Pad32(vOfs + ValLeng);
    // Now read the "Children". StringFileInfo and VarFileInfo have
    //   the same first few fields
    WHILE vOfs < VerLeng DO
      begin
        ChdBuff := @VerBuff^[vOfs DIV 2];
        ChdLeng := ChdBuff^[0]; // first WORD of XxxFileInfo
        IF fUnicode THEN // Key is UNICODE at offset 6
          begin
            ChdKey := WideCharToString(PWideChar(@ChdBuff^[3]));
            cOfs   := Pad32(6 + succ(length(ChdKey))*2);
          end
        ELSE            // Key is ANSI at offset 4
          begin
            ChdKey := StrPas(PChar(@ChdBuff^[2]));
            cOfs   := Pad32(4 + succ(length(ChdKey)));
          end;
        // Don't rely on wType field - check the key
        IF ChdKey = 'StringFileInfo' THEN
          begin
            WHILE cOfs < ChdLeng DO
              begin
                TabLeng := PWord(@ChdBuff^[cOfs DIV 2])^;
                VL := TVerLangObj.Create(@ChdBuff^[cOfs DIV 2],
                  fUnicode);
                IF (fStrLangs.Count > 0) AND (VL.Rating >
                  TVerLangObj(fStrLangs.Objects[0]).Rating) THEN
                  fStrLangs.InsertObject(0, VL.Language, VL)
                ELSE
                  fStrLangs.AddObject(VL.Language, VL);
                cOfs := pad32(cOfs + TabLeng);
              end;
          end
        ELSE IF ChdKey = 'VarFileInfo' THEN
          begin
            VarBuff := @ChdBuff^[cOfs DIV 2];
            VarLeng := VarBuff^[0]; // first WORD of XxxFileInfo
            IF fUnicode THEN // Key is UNICODE at offset 6
              begin
                VarKey := WideCharToString(PWideChar(@VarBuff^[3]));
                rOfs   := Pad32(6 + succ(length(VarKey))*2);
              end
            ELSE            // Key is ANSI at offset 4
              begin
                VarKey := StrPas(PChar(@VarBuff^[2]));
                rOfs   := Pad32(4 + succ(length(VarKey)));
              end;
            VarPs := @VarBuff^[rOfs DIV 2];
            WHILE rOfs < VarLeng DO
              begin
                fVarLangs.Add(Format('%.4x%.4x',
                  [LoWord(VarPs^), HiWord(VarPs^)]));
                //wha
                Inc(VarPs);
                Inc(rOfs, 4);
              end;
          end;
        // Set to offset of next child
        vOfs := Pad32(vOfs + ChdLeng);
      end;
    IF fStrLangs.Count = 0 THEN fStatus := vio_NoStrings
    ELSE fStatus := vio_OK;
  finally
    FreeMem(VerBuff);
  end;
end;

destructor TVerInfoObj.Destroy;
VAR N : Integer;
begin
  FOR N := 0 TO fStrLangs.Count-1 DO
    TVerLangObj(fStrLangs.Objects[N]).Free;
  fStrLangs.Free;
  inherited;
end;

function TVerInfoObj.GetCount: Integer;
begin
  IF fStrLangs.Count > 0 THEN
    Result := TVerLangObj(fStrLangs.objects[0]).fList.Count
  ELSE Result := 0;
end;

function TVerInfoObj.GetFileType: Integer;
begin
  Result := fVFFI.dwFileType;
end;

function TVerInfoObj.GetFileTypeS: String;
begin
  CASE fVFFI.dwFileType OF
    VFT_UNKNOWN    : Result := 'UNKNOWN';
    VFT_APP        : Result := 'APP';
    VFT_DLL        : Result := 'DLL';
    VFT_DRV        : Result := 'DRV';
    VFT_FONT       : Result := 'FONT';
    VFT_VXD        : Result := 'VXD';
    VFT_STATIC_LIB : Result := 'STATIC_LIB';
    ELSE             Result := '';
  END;
end;

function TVerInfoObj.GetFileVersion: String;
begin
  WITH fVFFI DO
    IF dwSignature = $FEEF04BD THEN
      Result := Format('%d.%d.%d.%d', [HiWord(dwFileVersionMS),
        LoWord(dwFileVersionMS), HiWord(dwFileVersionLS),
        LoWord(dwFileVersionLS)])
    ELSE Result := '';
end;

function TVerInfoObj.GetLangCount: Integer;
begin
  Result := fStrLangs.Count;
end;

function TVerInfoObj.GetLangString(IndexL, IndexS: Integer): string;
begin
  Result := TVerLangObj(fStrLangs.objects[IndexL]).fList[IndexS];
end;

function TVerInfoObj.GetLanguage(Index: Integer): string;
begin
  Result := fStrLangs[Index];
end;

function TVerInfoObj.GetLangValue(const Lang, Name: string): string;
VAR Idx : Integer;
begin
  Idx := fStrLangs.IndexOf(Lang);
  IF Idx = -1 THEN Result := ''
  ELSE Result := TVerLangObj(fStrLangs.objects[Idx]).GetValue(Name);
end;

function TVerInfoObj.GetProductVersion: String;
begin
  WITH fVFFI DO
    IF dwSignature = $FEEF04BD THEN
      Result := Format('%d.%d.%d.%d', [HiWord(dwProductVersionMS),
        LoWord(dwProductVersionMS), HiWord(dwProductVersionLS),
        LoWord(dwProductVersionLS)])
    ELSE Result := '';
end;

function TVerInfoObj.GetString(Index: Integer): string;
begin
  IF fStrLangs.Count > 0 THEN
    Result := TVerLangObj(fStrLangs.objects[0]).fList[Index]
  ELSE Result := '';
end;

function TVerInfoObj.GetValue(const Name: string): string;
begin
  IF fStrLangs.Count > 0 THEN
    Result := TVerLangObj(fStrLangs.objects[0]).GetValue(Name)
  ELSE Result := '';
end;

function TVerInfoObj.LangProb(var S: String): Boolean;
VAR
  N : Integer;
  S1 : String;
begin
  S  := '';
  S1 := '';
  IF fVarLangs.Count = 0 THEN
    S1 := '(NO VarFileInfo)'#13#10;
  FOR N := 0 TO fStrLangs.Count-1 DO
    IF fVarLangs.IndexOf(fStrLangs[N]) = -1 THEN
      S := S + fStrLangs[N] + ' ONLY StringFileInfo'#13#10
    ELSE S1 := S1 + fStrLangs[N] + ' (both - OK)'#13#10;
  FOR N := 0 TO fVarLangs.Count-1 DO
    IF fStrLangs.IndexOf(fVarLangs[N]) = -1 THEN
      S := S + fVarLangs[N] + ' ONLY VarFileInfo'#13#10;
  Result := S <> '';
  IF S <> '' THEN S := S + S1;
end;

{ TVerLangObj }

constructor TVerLangObj.Create(P: PByteArray; fUnicode : Boolean);
// P points to the start of a StringTable structure
VAR
  VerLeng  : DWORD;
  vOfs     : Word;
  ChdLeng  : DWORD;
  ChdKey   : String;
  ChdValue : String;
  ChdBuff  : PByteArray;
  cOfs     : Word;
begin
  fList     := TStringList.Create;
  fLanguage := '';
  VerLeng := PWord(P)^; // first WORD of StringTable structure
  IF fUnicode THEN // Key is UNICODE at offset 6
    begin
      fLanguage := WideCharToString(PWideChar(@P^[6]));
      vOfs := Pad32(6 + succ(length(fLanguage))*2);
    end
  ELSE            // Key is ANSI at offset 4
    begin
      fLanguage := StrPas(PChar(@P^[4]));
      vOfs := Pad32(4 + succ(length(fLanguage)));
    end;
  fLanguage := Uppercase(fLanguage);
  // Next come children - String structures
  WHILE vOfs < VerLeng DO
    begin
      ChdBuff := @P^[vOfs];
      ChdLeng := PWord(ChdBuff)^; // first WORD of String structure
      IF fUnicode THEN // Key is UNICODE at offset 6 -
        begin         //   UNICODE value follows
          ChdKey   := WideCharToString(PWideChar(@ChdBuff^[6]));
          cOfs     := Pad32(6 + succ(length(ChdKey))*2);
          ChdValue := WideCharToString(PWideChar(@ChdBuff^[cOfs]));
        end
      ELSE            // Key is ANSI at offset 4 -
        begin         //   ANSI value follows
          ChdKey   := StrPas(PChar(@ChdBuff^[4]));
          cOfs     := Pad32(4 + succ(length(ChdKey)));
          ChdValue := StrPas(PChar(@ChdBuff^[cOfs]));
        end;
      fList.Add(Format('%s=%s', [ChdKey, ChdValue]));
      vOfs := Pad32(vOfs + ChdLeng);
    end;
end;

destructor TVerLangObj.Destroy;
begin
  fList.Free;
  inherited;
end;

function TVerLangObj.GetValue(const Name: string): string;
begin
  Result := fList.Values[Name]
end;

function TVerLangObj.Rating: Integer;
begin
  IF fLanguage = PrefLang THEN
    Result := 3
  ELSE IF Copy(fLanguage, 1, 4) = Copy(PrefLang, 1, 4) THEN
    Result := 2
  ELSE Result := 1;
end;

VAR OVI : TOSVersionInfo;
initialization
  PrefLang := '040904E4';
  FillChar(OVI, SizeOf(OVI), 0);
  OVI.dwOSVersionInfoSize := SizeOf(OVI);
  IF GetVersionEx(OVI) THEN
    IsWinNT := OVI.dwPlatformId = VER_PLATFORM_WIN32_NT
  ELSE IsWinNT := False;
  IF IsWinNT THEN // Avoid stupid "never used" warning
    FillChar(OVI, SizeOf(OVI), 0);
end.
